<?php

namespace App\Services;

use App\Repository\Interface\OvertimeRepositoryInterface;
use App\DTOs\Overtime\OvertimeRequestFilterDTO;
use App\DTOs\Overtime\CreateOvertimeRequestDTO;
use App\DTOs\Overtime\UpdateOvertimeRequestDTO;
use App\DTOs\Overtime\OvertimeRequestResponseDTO;
use App\DTOs\Overtime\OvertimeStatsDTO;
use App\Models\User;
use App\Services\SimplePermissionService;
use App\Services\ApprovalService;
use App\Services\OvertimeCalculationService;
use App\Enums\StringStatusEnum;
use App\Enums\OvertimeReasonEnum;
use App\Enums\CompensationTypeEnum;
use App\Jobs\SendEmailNotificationJob;
use App\Mail\Overtime\OvertimeSubmitted;
use App\Mail\Overtime\OvertimeApproved;
use App\Mail\Overtime\OvertimeRejected;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;

class OvertimeService
{
    public function __construct(
        private readonly OvertimeRepositoryInterface $overtimeRepository,
        private readonly SimplePermissionService $permissionService,
        private readonly ApprovalService $approvalService,
        private readonly OvertimeCalculationService $calculationService,
        private readonly NotificationService $notificationService,
        private readonly ApprovalWorkflowService $approvalWorkflow,
        private readonly CacheService $cacheService
    ) {}

    /**
     * Get paginated overtime requests with permission filters.
     */
    public function getPaginatedRequests(OvertimeRequestFilterDTO $filters, User $user): array
    {
        $effectiveCompanyId = $this->permissionService->getEffectiveCompanyId($user);

        if ($user->user_type === 'company') {
            // Company users: get all requests, respect employee_id filter if provided
            $modifiedFilters = new OvertimeRequestFilterDTO(
                employeeId: $filters->employeeId, // Preserve employee_id from request
                status: $filters->status,
                overtimeReason: $filters->overtimeReason,
                fromDate: $filters->fromDate,
                toDate: $filters->toDate,
                month: $filters->month,
                search: $filters->search,
                companyId: $effectiveCompanyId,
                perPage: $filters->perPage,
                page: $filters->page
            );
        } else {
            // Staff users: check hierarchy permissions
            $canViewOthers = false;
            $subordinateIds = [];

            try {
                // Get all employees in the company except the user
                $allEmployees = User::where('company_id', $effectiveCompanyId)
                    ->where('user_id', '!=', $user->user_id)
                    ->get();

                foreach ($allEmployees as $employee) {
                    if ($this->permissionService->canViewEmployeeRequests($user, $employee)) {
                        $canViewOthers = true;
                        $subordinateIds[] = $employee->user_id;
                    }
                }

                // Always include the current user's own requests
                $subordinateIds[] = $user->user_id;
            } catch (\Exception $e) {
                $canViewOthers = false;
                $subordinateIds = [$user->user_id];
            }

            if ($canViewOthers && !empty($subordinateIds)) {
                // Manager: get requests for employees they can view
                // If employee_id is provided, verify it's in subordinates
                $effectiveEmployeeId = null;
                $effectiveEmployeeIds = $subordinateIds;

                if ($filters->employeeId !== null) {
                    $requestedEmployeeId = (int) $filters->employeeId;
                    Log::info('OvertimeService::getPaginatedRequests - Checking employee_id filter', [
                        'requested_employee_id' => $requestedEmployeeId,
                        'subordinate_ids' => $subordinateIds,
                        'in_array_result' => in_array($requestedEmployeeId, $subordinateIds, true),
                    ]);

                    if (in_array($requestedEmployeeId, $subordinateIds, true)) {
                        $effectiveEmployeeId = $requestedEmployeeId;
                        $effectiveEmployeeIds = null; // Use single employee filter
                    } else {
                        // Requested employee not in subordinates - use impossible ID to return empty
                        $effectiveEmployeeId = -1;
                        $effectiveEmployeeIds = null;
                    }
                }

                $modifiedFilters = new OvertimeRequestFilterDTO(
                    employeeId: $effectiveEmployeeId,
                    employeeIds: $effectiveEmployeeIds,
                    status: $filters->status,
                    overtimeReason: $filters->overtimeReason,
                    fromDate: $filters->fromDate,
                    toDate: $filters->toDate,
                    month: $filters->month,
                    search: $filters->search,
                    companyId: $effectiveCompanyId,
                    hierarchyLevels: null, // Don't apply hierarchy filtering by default
                    perPage: $filters->perPage,
                    page: $filters->page
                );
            } else {
                // Regular employee: only own requests
                $modifiedFilters = new OvertimeRequestFilterDTO(
                    employeeId: $user->user_id,
                    status: $filters->status,
                    overtimeReason: $filters->overtimeReason,
                    fromDate: $filters->fromDate,
                    toDate: $filters->toDate,
                    month: $filters->month,
                    search: $filters->search,
                    companyId: $effectiveCompanyId,
                    perPage: $filters->perPage,
                    page: $filters->page
                );
            }
        }

        $result = $this->overtimeRepository->getPaginatedRequests($modifiedFilters, $user);

        // Transform data using ResponseDTO for consistent format
        $transformedData = collect($result['data'])->map(function ($request) {
            return OvertimeRequestResponseDTO::fromModel($request)->toArray();
        })->toArray();

        return [
            'data' => $transformedData,
            'total' => $result['total'],
            'per_page' => $result['per_page'],
            'current_page' => $result['current_page'],
            'last_page' => $result['last_page'],
            'from' => $result['from'],
            'to' => $result['to'],
        ];
    }

    /**
     * Get overtime enums
     */
    public function getOvertimeEnums(): array
    {
        return [
            'statuses' => StringStatusEnum::toArray(),
            'reasons' => OvertimeReasonEnum::toArray(),
            'compensation_types' => CompensationTypeEnum::toArray(),
        ];
    }

    /**
     * Create overtime request with validation and calculations.
     */
    public function createRequest(CreateOvertimeRequestDTO $dto, User $user): OvertimeRequestResponseDTO
    {
        return DB::transaction(function () use ($dto, $user) {

            // Check hierarchical permissions for staff users creating requests for others
            if ($user->user_type !== 'company' && $dto->staffId !== $user->user_id) {
                $employee = User::find($dto->staffId);
                if (!$employee || !$this->permissionService->canViewEmployeeRequests($user, $employee)) {
                    Log::info('OvertimeService::createRequest - Permission denied', [
                        'user_id' => $user->user_id,
                        'staff_id' => $dto->staffId,
                        'message' => 'Permission denied'
                    ]);
                    throw new \Exception('ليس لديك صلاحية لإنشاء طلب عمل إضافي لهذا الموظف');
                }
            }

            // Check for overlapping overtime requests
            Log::info('Checking overtime overlap', [
                'staffId' => $dto->staffId,
                'requestDate' => $dto->requestDate,
                'clockIn' => $dto->clockIn,
                'clockOut' => $dto->clockOut,
                'message' => 'Checking overtime overlap'
            ]);

            if ($this->overtimeRepository->hasOverlappingOvertime($dto->staffId, $dto->requestDate, $dto->clockIn, $dto->clockOut)) {
                Log::info('Overtime overlap detected, throwing exception', [
                    'staffId' => $dto->staffId,
                    'requestDate' => $dto->requestDate,
                    'clockIn' => $dto->clockIn,
                    'clockOut' => $dto->clockOut,
                    'message' => 'Overtime overlap detected'
                ]);
                throw new \Exception('يوجد طلب عمل إضافي آخر لنفس الموظف في نفس الوقت أو وقت متداخل معه');
            }

            // Validate against shift
            $this->calculationService->validateAgainstShift(
                $dto->staffId,
                $dto->requestDate,
                $dto->clockIn,
                $dto->clockOut,
                $dto->overtimeReason
            );

            // Check if it's a holiday (allow for weekend work reason)
            if ($dto->overtimeReason !== OvertimeReasonEnum::SALARIED_EMPLOYEE->value) {
                $isHoliday = $this->calculationService->isHoliday($dto->companyId, $dto->requestDate);
                if ($isHoliday) {
                    Log::info('Holiday detected, throwing exception', [
                        'companyId' => $dto->companyId,
                        'requestDate' => $dto->requestDate,
                        'message' => 'Holiday detected'
                    ]);
                    throw new \Exception('لا يمكن تقديم طلب عمل إضافي في يوم عطلة');
                }
            }

            // Calculate total hours
            $totalHours = $this->calculationService->calculateTotalHours($dto->clockIn, $dto->clockOut);

            // Calculate overtime types
            $overtimeTypes = $this->calculationService->calculateOvertimeTypes(
                $totalHours,
                $dto->additionalWorkHours
            );

            // Calculate compensation banked
            $compensationBanked = $this->calculationService->calculateCompensationBanked(
                $totalHours,
                $dto->compensationType
            );

            // Prepare data for creation
            $data = [
                'company_id' => $dto->companyId,
                'staff_id' => $dto->staffId,
                'request_date' => $dto->requestDate,
                'request_month' => $dto->requestMonth,
                'clock_in' => $dto->clockIn,
                'clock_out' => $dto->clockOut,
                'overtime_reason' => $dto->overtimeReason,
                'additional_work_hours' => $dto->additionalWorkHours,
                'compensation_type' => $dto->compensationType,
                'request_reason' => $dto->requestReason,
                'straight' => $overtimeTypes['straight'],
                'time_a_half' => $overtimeTypes['time_a_half'],
                'double_overtime' => $overtimeTypes['double_overtime'],
                'total_hours' => $totalHours,
                'compensation_banked' => $compensationBanked,
                'is_approved' => 0, // Always pending
                'created_at' => date('d-m-Y H:i:s'),
            ];

            $request = $this->overtimeRepository->createRequest($data);

            // Start approval workflow if multi-level approval is enabled
            $this->approvalWorkflow->submitForApproval(
                'overtime_request_settings',
                (string)$request->time_request_id,
                $dto->companyId,
                $dto->staffId
            );
            // Send submission notification
            $notificationsSent = $this->notificationService->sendSubmissionNotification(
                'overtime_request_settings',
                (string)$request->time_request_id,
                $dto->companyId,
                StringStatusEnum::SUBMITTED->value,
                $dto->staffId
            );

            // If no notifications were sent (due to missing/invalid configuration),
            // send a notification to the employee as fallback
            if ($notificationsSent === 0 || $notificationsSent === null || !isset($notificationsSent)) {
                $this->notificationService->sendCustomNotification(
                    'overtime_request_settings',
                    (string)$request->time_request_id,
                    [$dto->staffId],
                    StringStatusEnum::SUBMITTED->value
                );
            }
            // Send email notification
            $employeeEmail = $request->employee->email ?? null;
            $employeeName = $request->employee->full_name ?? 'Employee';

            if ($employeeEmail) {
                SendEmailNotificationJob::dispatch(
                    new OvertimeSubmitted(
                        employeeName: $employeeName,
                        requestDate: $dto->requestDate,
                        totalHours: $totalHours,
                        reason: $dto->requestReason
                    ),
                    $employeeEmail
                );
            }

            Log::info('OvertimeService::createRequest completed', [
                'request_id' => $request->time_request_id
            ]);

            return OvertimeRequestResponseDTO::fromModel($request);
        });
    }

    /**
     * Update overtime request (only if pending and owner).
     */
    public function updateRequest(int $id, UpdateOvertimeRequestDTO $dto, User $user): OvertimeRequestResponseDTO
    {
        return DB::transaction(function () use ($id, $dto, $user) {
            $effectiveCompanyId = $this->permissionService->getEffectiveCompanyId($user);

            $request = $this->overtimeRepository->findRequestInCompany($id, $effectiveCompanyId);

            if (!$request) {
                Log::info('OvertimeService::updateRequest - Request not found', [
                    'request_id' => $id,
                    'message' => 'Request not found'
                ]);
                throw new \Exception('الطلب غير موجود');
            }

            // Check hierarchical permissions (owner, company, or authorized managers can update)
            $isOwner = $request->staff_id === $user->user_id;
            $isCompany = $user->user_type === 'company';

            if (!$isOwner && !$isCompany) {
                $employee = User::find($request->staff_id);
                if (!$employee || !$this->permissionService->canApproveEmployeeRequests($user, $employee)) {
                    Log::info('OvertimeService::updateRequest - Permission denied', [
                        'user_id' => $user->user_id,
                        'request_id' => $id,
                        'message' => 'Permission denied'
                    ]);
                    throw new \Exception('ليس لديك صلاحية لتعديل هذا الطلب');
                }
            }

            // Can only update pending requests
            if ($request->is_approved !== 0) {
                Log::info('OvertimeService::updateRequest - Request is not pending', [
                    'request_id' => $id,
                    'message' => 'Request is not pending'
                ]);
                throw new \Exception('لا يمكن تعديل طلب تمت مراجعته');
            }

            // Check for overlapping overtime requests (if times are being updated)
            $clockIn24 = $this->calculationService->convertTo24Hour($dto->clockIn, $dto->requestDate);
            $clockOut24 = $this->calculationService->convertTo24Hour($dto->clockOut, $dto->requestDate);

            if ($this->overtimeRepository->hasOverlappingOvertime($request->staff_id, $dto->requestDate, $clockIn24, $clockOut24, $id)) {
                Log::info('OvertimeService::updateRequest - Overlapping overtime detected', [
                    'request_id' => $id,
                    'message' => 'Overlapping overtime detected'
                ]);
                throw new \Exception('يوجد طلب عمل إضافي آخر لنفس الموظف في نفس الوقت أو وقت متداخل معه');
            }

            // Validate against shift
            $this->calculationService->validateAgainstShift(
                $request->staff_id,
                $dto->requestDate,
                $this->calculationService->convertTo24Hour($dto->clockIn, $dto->requestDate),
                $this->calculationService->convertTo24Hour($dto->clockOut, $dto->requestDate),
                $dto->overtimeReason
            );

            // Convert times
            $clockIn24 = $this->calculationService->convertTo24Hour($dto->clockIn, $dto->requestDate);
            $clockOut24 = $this->calculationService->convertTo24Hour($dto->clockOut, $dto->requestDate);

            // Calculate total hours
            $totalHours = $this->calculationService->calculateTotalHours($clockIn24, $clockOut24);

            // Calculate overtime types
            $overtimeTypes = $this->calculationService->calculateOvertimeTypes(
                $totalHours,
                $dto->additionalWorkHours
            );

            // Calculate compensation banked
            $compensationBanked = $this->calculationService->calculateCompensationBanked(
                $totalHours,
                $dto->compensationType
            );

            // Prepare update data
            $updateData = [
                'request_date' => $dto->requestDate,
                'request_month' => $this->calculationService->calculateRequestMonth($dto->requestDate),
                'clock_in' => $clockIn24,
                'clock_out' => $clockOut24,
                'overtime_reason' => $dto->overtimeReason,
                'additional_work_hours' => $dto->additionalWorkHours,
                'compensation_type' => $dto->compensationType,
                'request_reason' => $dto->requestReason,
                'straight' => $overtimeTypes['straight'],
                'time_a_half' => $overtimeTypes['time_a_half'],
                'double_overtime' => $overtimeTypes['double_overtime'],
                'total_hours' => $totalHours,
                'compensation_banked' => $compensationBanked,
            ];

            $updatedRequest = $this->overtimeRepository->updateRequest($request, $updateData);

            return OvertimeRequestResponseDTO::fromModel($updatedRequest);
        });
    }

    /**
     * Delete overtime request (only if pending and owner).
     */
    public function deleteRequest(int $id, User $user): bool
    {
        $effectiveCompanyId = $this->permissionService->getEffectiveCompanyId($user);

        $request = $this->overtimeRepository->findRequestInCompany($id, $effectiveCompanyId);

        if (!$request) {
            Log::info('OvertimeService::deleteRequest - Request not found', [
                'request_id' => $id,
                'message' => 'Request not found'
            ]);
            throw new \Exception('الطلب غير موجود');
        }

        // Check hierarchical permissions (owner, company, or authorized managers can delete)
        $isOwner = $request->staff_id === $user->user_id;
        $isCompany = $user->user_type === 'company';

        if (!$isOwner && !$isCompany) {
            $employee = User::find($request->staff_id);
            if (!$employee || !$this->permissionService->canViewEmployeeRequests($user, $employee)) {
                Log::info('OvertimeService::deleteRequest - Permission denied', [
                    'user_id' => $user->user_id,
                    'request_id' => $id,
                    'message' => 'Permission denied'
                ]);
                throw new \Exception('ليس لديك صلاحية لحذف هذا الطلب');
            }
        }

        // Can only delete pending requests
        if ($request->is_approved !== 0) {
            Log::info('OvertimeService::deleteRequest - Request is not pending', [
                'request_id' => $id,
                'message' => 'Request is not pending'
            ]);
            throw new \Exception('لا يمكن حذف طلب تمت مراجعته');
        }

        // Determine cancel reason based on who is cancelling
        $cancelReason = $isOwner
            ? 'تم إلغاء الطلب من قبل الموظف'
            : 'تم إلغاء الطلب من قبل الإدارة';

        $this->overtimeRepository->rejectRequest($request, $user->user_id, $cancelReason);

        return true;
    }

    /**
     * Approve overtime request with multi-level approval workflow.
     */
    public function approveRequest(int $id, User $approver, ?string $remarks = null): OvertimeRequestResponseDTO
    {
        return DB::transaction(function () use ($id, $approver, $remarks) {
            $effectiveCompanyId = $this->permissionService->getEffectiveCompanyId($approver);

            $request = $this->overtimeRepository->findRequestInCompany($id, $effectiveCompanyId);

            if (!$request) {
                Log::info('OvertimeService::approveRequest - Request not found', [
                    'request_id' => $id,
                    'message' => 'Request not found'
                ]);
                throw new \Exception('الطلب غير موجود');
            }

            if ($request->is_approved !== 0) {
                Log::info('OvertimeService::approveRequest - Request is not pending', [
                    'request_id' => $id,
                    'message' => 'Request is not pending'
                ]);
                throw new \Exception('تمت مراجعة هذا الطلب مسبقاً');
            }

            // Check hierarchy permissions for staff users (strict: must be higher level)
            if ($approver->user_type !== 'company') {
                $employee = User::find($request->staff_id);
                if (!$employee || !$this->permissionService->canApproveEmployeeRequests($approver, $employee)) {
                    Log::warning('OvertimeService::approveRequest - Hierarchy permission denied', [
                        'request_id' => $id,
                        'requester_id' => $approver->user_id,
                        'requester_type' => $approver->user_type,
                        'employee_id' => $request->staff_id,
                        'requester_level' => $this->permissionService->getUserHierarchyLevel($approver),
                        'employee_level' => $this->permissionService->getUserHierarchyLevel($employee),
                        'requester_department' => $this->permissionService->getUserDepartmentId($approver),
                        'employee_department' => $this->permissionService->getUserDepartmentId($employee),
                        'message' => 'Hierarchy permission denied'
                    ]);
                    throw new \Exception('ليس لديك صلاحية للموافقة على طلب هذا الموظف');
                }
            }

            $userType = strtolower(trim($approver->user_type ?? ''));

            // Company user can approve directly
            if ($userType === 'company') {
                $approvedRequest = $this->overtimeRepository->approveRequest($request, $approver->user_id);

                // Send approval notification
                $this->notificationService->sendApprovalNotification(
                    'overtime_request_settings',
                    (string)$request->time_request_id,
                    $effectiveCompanyId,
                    StringStatusEnum::APPROVED->value,
                    $approver->user_id,
                    1,
                    $request->staff_id
                );

                // Send email notification
                $employeeEmail = $request->employee->email ?? null;
                $employeeName = $request->employee->full_name ?? 'Employee';

                if ($employeeEmail) {
                    SendEmailNotificationJob::dispatch(
                        new OvertimeApproved(
                            employeeName: $employeeName,
                            requestDate: $request->request_date,
                            totalHours: $request->total_hours,
                            remarks: $remarks
                        ),
                        $employeeEmail
                    );
                }

                // Record final approval
                $this->approvalService->recordApproval(
                    $request->time_request_id,
                    $approver->user_id,
                    1, // approved
                    1, // final level
                    'overtime_request_settings',
                    $effectiveCompanyId,
                    $request->staff_id
                );

                return OvertimeRequestResponseDTO::fromModel($approvedRequest);
            }

            // For staff users, use multi-level approval
            $canApprove = $this->approvalService->canUserApprove(
                $approver->user_id,
                $request->time_request_id,
                $request->staff_id,
                'overtime_request_settings'
            );

            if (!$canApprove) {
                $denialInfo = $this->approvalService->getApprovalDenialReason(
                    $approver->user_id,
                    $request->time_request_id,
                    $request->staff_id,
                    'overtime_request_settings'
                );
                Log::info('OvertimeService::approveRequest - Permission denied', [
                    'user_id' => $approver->user_id,
                    'request_id' => $id,
                    'message' => $denialInfo['message']
                ]);
                throw new \Exception($denialInfo['message']);
            }

            $isFinal = $this->approvalService->isFinalApproval(
                $request->time_request_id,
                $request->staff_id,
                'overtime_request_settings'
            );

            if ($isFinal) {
                // Final approval - update request status
                $approvedRequest = $this->overtimeRepository->approveRequest($request, $approver->user_id);

                // Send approval notification
                $this->notificationService->sendApprovalNotification(
                    'overtime_request_settings',
                    (string)$request->time_request_id,
                    $effectiveCompanyId,
                    StringStatusEnum::APPROVED->value,
                    $approver->user_id,
                    null,
                    $request->staff_id
                );

                // Record final approval
                $this->approvalService->recordApproval(
                    $request->time_request_id,
                    $approver->user_id,
                    1, // approved
                    1, // final level
                    'overtime_request_settings',
                    $effectiveCompanyId,
                    $request->staff_id
                );

                // Send approval email
                $employeeEmail = $approvedRequest->employee->email ?? null;
                $employeeName = $approvedRequest->employee->full_name ?? 'Employee';

                if ($employeeEmail) {
                    SendEmailNotificationJob::dispatch(
                        new OvertimeApproved(
                            employeeName: $employeeName,
                            requestDate: $approvedRequest->request_date,
                            totalHours: $approvedRequest->total_hours,
                            remarks: $remarks
                        ),
                        $employeeEmail
                    );
                }

                return OvertimeRequestResponseDTO::fromModel($approvedRequest);
            } else {
                // Intermediate approval - just record it
                $this->approvalService->recordApproval(
                    $request->time_request_id,
                    $approver->user_id,
                    1, // approved
                    0, // intermediate level
                    'overtime_request_settings',
                    $effectiveCompanyId,
                    $request->staff_id
                );

                // Send approval notification
                $this->notificationService->sendApprovalNotification(
                    'overtime_request_settings',
                    (string)$request->time_request_id,
                    $effectiveCompanyId,
                    StringStatusEnum::APPROVED->value,
                    $approver->user_id,
                    1,
                    $request->staff_id
                );
                // Reload to get updated approvals
                $request->refresh();
                $request->load(['employee', 'approvals.staff']);

                return OvertimeRequestResponseDTO::fromModel($request);
            }
        });
    }

    /**
     * Reject overtime request.
     */
    public function rejectRequest(int $id, User $rejector, string $reason): OvertimeRequestResponseDTO
    {
        return DB::transaction(function () use ($id, $rejector, $reason) {

            $effectiveCompanyId = $this->permissionService->getEffectiveCompanyId($rejector);
            $request = $this->overtimeRepository->findRequestInCompany($id, $effectiveCompanyId);

            if (!$request) {
                Log::info('OvertimeService::rejectRequest - Request not found', [
                    'id' => $id,
                    'message' => 'الطلب غير موجود'
                ]);
                throw new \Exception('الطلب غير موجود');
            }

            if ($request->is_approved !== 0) {
                Log::info('OvertimeService::rejectRequest - Request is not pending', [
                    'id' => $id,
                    'message' => 'تمت مراجعة هذا الطلب مسبقاً'
                ]);
                throw new \Exception('تمت مراجعة هذا الطلب مسبقاً');
            }

            // Check hierarchy permissions for staff users (strict: must be higher level)
            if ($rejector->user_type !== 'company') {
                $employee = User::find($request->staff_id);
                if (!$employee || !$this->permissionService->canApproveEmployeeRequests($rejector, $employee)) {
                    Log::warning('OvertimeService::rejectRequest - Hierarchy permission denied', [
                        'request_id' => $id,
                        'requester_id' => $rejector->user_id,
                        'requester_type' => $rejector->user_type,
                        'employee_id' => $request->staff_id,
                        'requester_level' => $this->permissionService->getUserHierarchyLevel($rejector),
                        'employee_level' => $this->permissionService->getUserHierarchyLevel($employee),
                        'requester_department' => $this->permissionService->getUserDepartmentId($rejector),
                        'employee_department' => $this->permissionService->getUserDepartmentId($employee),
                        'message' => 'ليس لديك صلاحية لرفض طلب هذا الموظف'
                    ]);
                    throw new \Exception('ليس لديك صلاحية لرفض طلب هذا الموظف');
                }
            }

            $userType = strtolower(trim($rejector->user_type ?? ''));

            if ($userType !== 'company') {
                $canApprove = $this->approvalService->canUserApprove(
                    $rejector->user_id,
                    $request->time_request_id,
                    $request->staff_id,
                    'overtime_request_settings'
                );
                if (!$canApprove) {
                    Log::warning('OvertimeService::rejectRequest - Company permission denied', [
                        'request_id' => $id,
                        'requester_id' => $rejector->user_id,
                        'requester_type' => $rejector->user_type,
                        'employee_id' => $request->staff_id,
                        'message' => 'ليس لديك صلاحية لرفض طلب هذا الموظف'
                    ]);
                    throw new \Exception('ليس لديك صلاحية لرفض طلب هذا الموظف');
                }
            }
            // Reject the request
            $rejectedRequest = $this->overtimeRepository->rejectRequest($request, $rejector->user_id, $reason);

            if (!$rejectedRequest) {
                Log::info('OvertimeService::rejectRequest - Request not found', [
                    'id' => $id,
                    'message' => 'فشل في رفض الطلب'
                ]);
                throw new \Exception('فشل في رفض الطلب');
            }

            // Send rejection notification
            $this->notificationService->sendApprovalNotification(
                'overtime_request_settings',
                (string)$request->time_request_id,
                $effectiveCompanyId,
                StringStatusEnum::REJECTED->value,
                $rejector->user_id,
                null,
                $request->staff_id
            );

            // Send email notification
            $employeeEmail = $request->employee->email ?? null;
            $employeeName = $request->employee->full_name ?? 'Employee';

            if ($employeeEmail) {
                SendEmailNotificationJob::dispatch(
                    new OvertimeRejected(
                        employeeName: $employeeName,
                        requestDate: $request->request_date,
                        totalHours: $request->total_hours,
                        reason: $reason
                    ),
                    $employeeEmail
                );
            }

            // Record rejection
            $this->approvalService->recordApproval(
                $request->time_request_id,
                $rejector->user_id,
                2, // rejected
                2, // rejection level
                'overtime_request_settings',
                $effectiveCompanyId,
                $request->staff_id
            );

            return OvertimeRequestResponseDTO::fromModel($rejectedRequest);
        });
    }

    /**
     * Get requests requiring approval from specific user.
     */
    public function getRequestsForApproval(User $user): array
    {
        $effectiveCompanyId = $this->permissionService->getEffectiveCompanyId($user);

        $requests = $this->overtimeRepository->getRequestsRequiringApproval(
            $user->user_id,
            $effectiveCompanyId
        );

        return array_map(
            fn($request) => OvertimeRequestResponseDTO::fromModel($request)->toArray(),
            $requests->all()
        );
    }

    /**
     * Get team requests (for managers).
     */
    public function getTeamRequests(User $manager): array
    {
        $effectiveCompanyId = $this->permissionService->getEffectiveCompanyId($manager);

        $requests = $this->overtimeRepository->getRequestsByManager(
            $manager->user_id,
            $effectiveCompanyId
        );

        return array_map(
            fn($request) => OvertimeRequestResponseDTO::fromModel($request)->toArray(),
            $requests->all()
        );
    }

    /**
     * Get overtime statistics.
     */
    public function getStats(User $user, ?string $fromDate = null, ?string $toDate = null): OvertimeStatsDTO
    {
        $effectiveCompanyId = $this->permissionService->getEffectiveCompanyId($user);

        $stats = $this->overtimeRepository->getStats($effectiveCompanyId, $fromDate, $toDate);

        return OvertimeStatsDTO::fromData($stats);
    }

    /**
     * Get single overtime request.
     */
    public function getRequest(int $id, User $user): OvertimeRequestResponseDTO
    {
        $effectiveCompanyId = $this->permissionService->getEffectiveCompanyId($user);

        $request = $this->overtimeRepository->findRequestInCompany($id, $effectiveCompanyId);

        if (!$request) {
            Log::info('OvertimeService::getRequest - Request not found', [
                'id' => $id,
                'message' => 'Request not found'
            ]);
            throw new \Exception('الطلب غير موجود');
        }

        // Check hierarchical permissions
        if ($user->user_type !== 'company' && $request->staff_id !== $user->user_id) {
            $employee = User::find($request->staff_id);
            if (!$employee || !$this->permissionService->canViewEmployeeRequests($user, $employee)) {
                Log::warning('OvertimeService::getRequest - Hierarchy permission denied', [
                    'request_id' => $id,
                    'requester_id' => $user->user_id,
                    'requester_type' => $user->user_type,
                    'employee_id' => $request->staff_id,
                    'requester_level' => $this->permissionService->getUserHierarchyLevel($user),
                    'employee_level' => $this->permissionService->getUserHierarchyLevel($employee),
                    'requester_department' => $this->permissionService->getUserDepartmentId($user),
                    'employee_department' => $this->permissionService->getUserDepartmentId($employee),
                    'message' => 'Hierarchy permission denied'
                ]);
                throw new \Exception('ليس لديك صلاحية لعرض هذا الطلب');
            }
        }

        return OvertimeRequestResponseDTO::fromModel($request);
    }
}
